<?php

namespace PartnerAdsWooCommerce\Feed;

use PartnerAdsWooCommerce\Feed\Mapping\XmlMapper;
use PartnerAdsWooCommerce\Settings\Setting;

class Generator
{
    /**
     * Generate & Write the feed to a file
     * @return bool
     */
    public function writeFeed()
    {
        $generating_feed = get_option('partner-ads-generating_feed', false);
        if ($generating_feed) {
            $create_new_file = false;
        } else {
            $generating_feed = [
                'count_products' => wp_count_posts('product', 'product_variation'),
                'da' => [
                    'post__not_in' => [],
                    'finished' => false
                ],
                'no' => [
                    'post__not_in' => [],
                    'finished' => false
                ],
                'sv' => [
                    'post__not_in' => [],
                    'finished' => false
                ],
                'se' => [
                    'post__not_in' => [],
                    'finished' => false
                ],
            ];
            $create_new_file = true;
            update_option('partner-ads-generating_feed', $generating_feed, false);
        }

        if (function_exists('wpml_language_is_active_filter')) {
            global $sitepress;
            $amount_per_batch = 100;

            $langauges = ['da', 'no', 'sv', 'se'];
            $still_have_products = false;
            foreach ($langauges as $language) {
                if (
                    wpml_language_is_active_filter('', $language)
                    ||
                    Setting::get('fallback_country') == $language
                ) {
                    if ($generating_feed[$language]['finished']) {
                        continue;
                    }

                    $sitepress->switch_lang($language);

                    add_filter('woocommerce_currency', function ($currency) use ($language) {
                        if($language == 'da') {
                            return 'DKK';
                        } elseif($language == 'no') {
                            return 'NOK';
                        } elseif($language == 'sv' || $language == 'se') {
                            return 'SEK';
                        }

                        return $currency;
                    });

                    $productsClass = new Products();
                    $products = $productsClass->getProducts($generating_feed[$language]['post__not_in'], $amount_per_batch, $language);
                    if ($products) {
                        $mapping = new XmlMapper($products);
                        $productXml = $mapping->map($create_new_file);
                        $this->saveToFile($productXml, $language . '_feed', $create_new_file);
                        $still_have_products = true;
                    } else {
                        if (count($generating_feed[$language]['post__not_in'])) {
                            $this->saveToFile('</products></xml>', $language . '_feed', $create_new_file);
                        }
                        $generating_feed[$language]['finished'] = true;
                        update_option('partner-ads-generating_feed', $generating_feed, false);
                    }
                }
            }

            if ($still_have_products) {
                if (!wp_next_scheduled('partner_ads_batch_generate_feed')) {
                    wp_schedule_single_event(time() + 2, 'partner_ads_batch_generate_feed');
                }
            } else {
                delete_option('partner-ads-generating_feed');
            }
        } else {
            $productsClass = new Products();
            $amount_per_batch = 250;

            if(!isset($generating_feed['dk']['post__not_in'])) {
                $generating_feed['dk']['post__not_in'] = [];
            }
            $products = $productsClass->getProducts($generating_feed['dk']['post__not_in'], $amount_per_batch, 'dk');
            if ($products) {
                $mapping = new XmlMapper($products);
                $productXml = $mapping->map($create_new_file);
                $this->saveToFile($productXml, 'feed', $create_new_file);
                if (!wp_next_scheduled('partner_ads_batch_generate_feed')) {
                    wp_schedule_single_event(time() + 2, 'partner_ads_batch_generate_feed');
                }
            } else {
                $this->saveToFile('</products></xml>', 'feed', false);
                delete_option('partner-ads-generating_feed');
            }
        }
    }

    /**
     * Save to the file
     * @param $content
     * @return bool
     */
    public function saveToFile($content, $key, $empty)
    {
        $upload_dir = wp_upload_dir();
        $baseUrl = $upload_dir['baseurl'];
        if (is_ssl() && preg_match('/http:\/\//', $baseUrl)) {
            $baseUrl = str_replace('http://', 'https://', $baseUrl);
        }
        $directory = '/partner-ads-feed/';
        $path = $upload_dir['basedir'] . $directory;
        $fileName = $key . '.xml';
        $filePath = $path . $fileName;
        $fileUrl = $baseUrl . $directory . $fileName;

        if ($this->checkDirectory($path)) {
            if ($empty) {
                file_put_contents($filePath, '');
            }

            file_put_contents($filePath, $content, FILE_APPEND);

            Setting::update($key, $fileUrl);
            Setting::update('feed_last_update', (new \DateTime('NOW'))->format('Y-m-d H:i:s'));

            return true;
        }

        return false;
    }

    /**
     * Make sure our Directory Exist
     * @param $path
     * @return bool
     */
    private function checkDirectory($path)
    {
        if (!file_exists($path)) {
            return wp_mkdir_p($path);
        }

        return true;
    }
}
