<?php

namespace PartnerAdsWooCommerce\Feed;

use PartnerAdsWooCommerce\Settings\Setting;

class Products
{
    /**
     * Contain all parent product information for the plugin.
     *
     * @access   protected
     * @var      string $parent Contain all parent product information for the plugin.
     */
    public $parent;

    /**
     * Contain all child product information for the plugin.
     *
     * @access   protected
     * @var      string $parent Contain all child product information for the plugin.
     */
    public $child;

    /**
     * The parent id of current product.
     *
     * @access   private
     * @var      string $parentID The current product's Parent ID.
     */
    public $parentID;
    /**
     * The child id of current product.
     *
     * @access   private
     * @var      string $parentID The current product's child ID.
     */
    public $childID;

    /**
     * The Variable that contain all products.
     *
     * @access   private
     * @var      array $productsList Products list array.
     */
    public $productsList;

    /**
     * The Variable that contain all attributes.
     *
     * @access   private
     * @var      array $attributeList attributes list array.
     */
    public $attributeList;

    /**
     * The string of the Brand Taxonomy
     *
     * @var string
     */
    private $brandTax;

    /**
     * The string of the Custom Brand Attribute
     *
     * @var string
     */
    private $brandTaxCustom;

    /**
     * The string of the Color Taxonomy
     *
     * @var string
     */
    private $colorTax;

    /**
     * The string of the Custom Color Attribute
     *
     * @var string
     */
    private $colorTaxCustom;

    /**
     * The string of the Size Taxonomy
     *
     * @var string
     */
    private $sizeTax;

    /**
     * The string of the Custom Size Attribute
     *
     * @var string
     */
    private $sizeTaxCustom;

    /**
     * The string of the Gender Taxonomy
     *
     * @var string
     */
    private $genderTax;

    /**
     * The string of the Custom Gender Attribute
     *
     * @var string
     */
    private $genderTaxCustom;

    /**
     * Get WooCommerce Products
     * @return array
     */
    public function getProducts($post__not_in = [], $posts_per_page = 500, $language = 'dk')
    {
        // Query Database for products
        $query = apply_filters('partner-ads-woocommerce-feed-product-query', [
            'post_type' => ['product'],
            'post_status' => 'publish',
            'posts_per_page' => $posts_per_page,
            'post__not_in' => $post__not_in,
        ]);
        $loop = new \WP_Query($query);

        $i = 0;

        $this->brandTax = Setting::get('feed_brand');
        if ($this->brandTax == 'custom_attribute') {
            $this->brandTaxCustom = Setting::get('feed_brand_custom');
        }

        $this->colorTax = Setting::get('feed_color');
        if ($this->colorTax == 'custom_attribute') {
            $this->colorTaxCustom = Setting::get('feed_color_custom');
        }

        $this->sizeTax = Setting::get('feed_size');
        if ($this->sizeTax == 'custom_attribute') {
            $this->sizeTaxCustom = Setting::get('feed_size_custom');
        }

        $this->genderTax = Setting::get('feed_gender');
        if ($this->genderTax == 'custom_attribute') {
            $this->genderTaxCustom = Setting::get('feed_gender_custom');
        }

        while ($loop->have_posts()) : $loop->the_post();
            $this->childID = get_the_ID();
            $post__not_in[] = $this->childID;
            $this->parentID = wp_get_post_parent_id($this->childID);

            global $product, $wpdb;
            if (!is_object($product)) {
                continue;
            }

            if (get_post_meta($product->get_id(), 'exclude_partner_ads_feed', true) == '1') {
                continue;
            }

            $type = '';
            if (is_object($product) && $product->is_type('simple')) {
                // No variations to product
                $type = 'simple';
            } elseif (is_object($product) && $product->is_type('variable')) {
                // Product has variations
                $type = 'variable';
            } elseif (is_object($product) && $product->is_type('grouped')) {
                $type = 'grouped';
            } elseif (is_object($product) && $product->is_type('external')) {
                $type = 'external';
            } elseif (is_object($product) && $product->is_downloadable()) {
                $type = 'downloadable';
            } elseif (is_object($product) && $product->is_virtual()) {
                $type = 'virtual';
            }

            if ($type == 'variable' && Setting::get('feed_add_variants') == 'yes') {
                $children = $product->get_visible_children();
                foreach ($children as $child) {
                    $child_product = wc_get_product($child);
                    if ($child_product->get_stock_status() != 'instock' && Setting::get('feed_include_out_of_stock') != 'yes') {
                        continue;
                    }

                    if (
                        ($this->get_regular_price($child_product) == 0 || $this->get_regular_price($child_product) == '')
                        && Setting::get('feed_include_zero_prices') != 'yes'
                    ) {
                        continue;
                    }

                    $this->add_variable_product_data($child_product, $type, $i, $language, $product);
                    $i++;
                }
            } else {
                if ($product->get_stock_status() != 'instock' && Setting::get('feed_include_out_of_stock') != 'yes') {
                    continue;
                }

                if (
                    ($this->get_regular_price($product) == 0 || $this->get_regular_price($product) == '')
                    && Setting::get('feed_include_zero_prices') != 'yes'
                ) {
                    continue;
                }

                $this->add_product_data($product, $type, $i, $language);
            }

            $i++;
        endwhile;
        wp_reset_query();

        $generating_feed = get_option('partner-ads-generating_feed', false);
        $generating_feed[$language]['post__not_in'] = $post__not_in;
        update_option('partner-ads-generating_feed', $generating_feed);

        return apply_filters('partner-ads-woocommerce-feed-product-link', $this->productsList);
    }

    public function add_product_data($product, $type, $i, $language)
    {
        $post = get_post($this->parentID);
        $this->set_common_product_data($product, $i, $language);

        $attributes = $product->get_attributes();
        if ($this->brandTax) {
            if ($this->brandTax == 'custom_attribute') {
                $attributes = get_post_meta($product->get_id(), '_product_attributes', true);
                if ($attributes) {
                    foreach ($attributes as $attribute_type) {
                        if ($attribute_type['name'] == $this->brandTaxCustom) {
                            $this->productsList[$i]['brand'] = $attribute_type['value'];
                        }
                    }
                }
            } elseif ($this->brandTax == 'perfect_brands_for_woocommerce') {
                $terms = wp_get_object_terms($post->ID, 'pwb-brand');
                $names = wp_list_pluck($terms, 'name');
                $this->productsList[$i]['brand'] = implode('|', $names);
            } elseif ($this->brandTax == 'brand_tax') {
                $terms = wp_get_object_terms($post->ID, 'product_brand');
                if (!$terms) {
                    $terms = wp_get_object_terms($post->ID, 'brand');
                }
                $names = wp_list_pluck($terms, 'name');
                $this->productsList[$i]['brand'] = implode('|', $names);
            } elseif (isset($attributes[$this->brandTax])) {
                if (is_a($attributes[$this->brandTax], 'WC_Product_Attribute')) {
                    $terms = $attributes[$this->brandTax]->get_terms();
                } else {
                    $terms = wp_get_object_terms($post->ID, $this->brandTax);
                }
                $names = wp_list_pluck($terms, 'name');
                $this->productsList[$i]['brand'] = implode('|', $names);
            }
        }

        if ($this->sizeTax) {
            if ($this->sizeTax == 'custom_attribute') {
                $attributes = get_post_meta($product->get_id(), '_product_attributes', true);
                if ($attributes) {
                    foreach ($attributes as $attribute_type) {
                        if ($attribute_type['name'] == $this->sizeTaxCustom) {
                            $this->productsList[$i]['sizes'] = $attribute_type['value'];
                        }
                    }
                }
            } elseif (isset($attributes[$this->sizeTax])) {
                if (is_a($attributes[$this->sizeTax], 'WC_Product_Attribute')) {
                    $terms = $attributes[$this->sizeTax]->get_terms();
                } else {
                    $terms = wp_get_object_terms($post->ID, $this->sizeTax);
                }
                $names = wp_list_pluck($terms, 'name');
                $this->productsList[$i]['sizes'] = implode('|', $names);
            }
        }

        if ($this->genderTax) {
            if ($this->genderTax == 'custom_attribute') {
                $attributes = get_post_meta($product->get_id(), '_product_attributes', true);
                if ($attributes) {
                    foreach ($attributes as $attribute_type) {
                        if ($attribute_type['name'] == $this->genderTaxCustom) {
                            $this->productsList[$i]['gender'] = $attribute_type['value'];
                        }
                    }
                }
            } elseif (isset($attributes[$this->genderTax])) {
                if (is_a($attributes[$this->genderTax], 'WC_Product_Attribute')) {
                    $terms = $attributes[$this->genderTax]->get_terms();
                } else {
                    $terms = wp_get_object_terms($post->ID, $this->genderTax);
                }
                $names = wp_list_pluck($terms, 'name');
                $this->productsList[$i]['gender'] = implode('|', $names);
            }
        }

        if ($this->colorTax) {
            if ($this->colorTax == 'custom_attribute') {
                $attributes = get_post_meta($product->get_id(), '_product_attributes', true);
                if ($attributes) {
                    foreach ($attributes as $attribute_type) {
                        if ($attribute_type['name'] == $this->colorTaxCustom) {
                            $this->productsList[$i]['colors'] = $attribute_type['value'];
                        }
                    }
                }
            } elseif (isset($attributes[$this->colorTax])) {
                if (is_a($attributes[$this->colorTax], 'WC_Product_Attribute')) {
                    $terms = $attributes[$this->colorTax]->get_terms();
                } else {
                    $terms = wp_get_object_terms($post->ID, $this->colorTax);
                }
                $names = wp_list_pluck($terms, 'name');
                $this->productsList[$i]['colors'] = implode('|', $names);
            }
        }

        /**
         * Overwrite some settings with special products
         */
        if ($type == 'external') {
            $getLink = new \WC_Product_External($post->ID);
            $link = $getLink->get_product_url();
            $this->productsList[$i]['link'] = $link;
        }
    }

    public function get_attributes()
    {
    }

    public function add_variable_product_data($product, $type, $i, $language, $parent_product)
    {
        $post = get_post($this->parentID);
        $this->set_common_product_data($product, $i, $language);

        $attributes = $product->get_attributes();
        $variations_attributes = $product->get_variation_attributes();
        $parent_attributes = $parent_product->get_attributes();

        if ($this->brandTax) {
            if (in_array('attribute_'. $this->brandTax, array_keys($variations_attributes))) {
                $term_slug = $variations_attributes['attribute_'. $this->brandTax];
                $term = get_term_by('slug', $term_slug, $this->brandTax);
                if ($term) {
                    $this->productsList[$i]['brand'] = $term->name;
                    $this->productsList[$i]['title'] .= ' - '. $term->name;
                }
            } elseif ($this->brandTax == 'perfect_brands_for_woocommerce') {
                $terms = wp_get_object_terms($post->ID, 'pwb-brand');
                $names = wp_list_pluck($terms, 'name');
                $this->productsList[$i]['brand'] = implode('|', $names);
            } elseif ($this->brandTax == 'brand_tax') {
                $terms = wp_get_object_terms($post->ID, 'brand');
                $names = wp_list_pluck($terms, 'name');
                $this->productsList[$i]['brand'] = implode('|', $names);
            } elseif (isset($attributes[$this->brandTax])) {
                if (is_a($attributes[$this->brandTax], 'WC_Product_Attribute')) {
                    $terms = $attributes[$this->brandTax]->get_terms();
                } else {
                    $terms = wp_get_object_terms($post->ID, $this->brandTax);
                }
                $names = wp_list_pluck($terms, 'name');
                $this->productsList[$i]['brand'] = implode('|', $names);
            } elseif (isset($parent_attributes[$this->brandTax])) {
                if (is_a($parent_attributes[$this->brandTax], 'WC_Product_Attribute')) {
                    $terms = $parent_attributes[$this->brandTax]->get_terms();
                } else {
                    $terms = wp_get_object_terms($post->ID, $this->brandTax);
                }
                $names = wp_list_pluck($terms, 'name');
                $this->productsList[$i]['brand'] = implode('|', $names);
            }
        }

        if ($this->sizeTax) {
            if (in_array('attribute_'. $this->sizeTax, array_keys($variations_attributes))) {
                $term_slug = $variations_attributes['attribute_'. $this->sizeTax];
                $term = get_term_by('slug', $term_slug, $this->sizeTax);

                if ($term) {
                    $this->productsList[$i]['size'] = $term->name;
                    $this->productsList[$i]['title'] .= ' - '. $term->name;
                }
            } elseif (isset($attributes[$this->sizeTax])) {
                if (is_a($attributes[$this->sizeTax], 'WC_Product_Attribute')) {
                    $terms = $attributes[$this->sizeTax]->get_terms();
                } else {
                    $terms = wp_get_object_terms($post->ID, $this->sizeTax);
                }
                $names = wp_list_pluck($terms, 'name');
                $this->productsList[$i]['size'] = implode('|', $names);
            } elseif (isset($parent_attributes[$this->sizeTax])) {
                if (is_a($parent_attributes[$this->sizeTax], 'WC_Product_Attribute')) {
                    $terms = $parent_attributes[$this->sizeTax]->get_terms();
                } else {
                    $terms = wp_get_object_terms($post->ID, $this->sizeTax);
                }
                $names = wp_list_pluck($terms, 'name');
                $this->productsList[$i]['size'] = implode('|', $names);
            }
        }

        if ($this->genderTax) {
            if (in_array('attribute_'. $this->genderTax, array_keys($variations_attributes))) {
                $term_slug = $variations_attributes['attribute_'. $this->genderTax];
                $term = get_term_by('slug', $term_slug, $this->genderTax);

                if ($term) {
                    $this->productsList[$i]['gender'] = $term->name;
                    $this->productsList[$i]['title'] .= ' - '. $term->name;
                }
            } elseif (isset($attributes[$this->genderTax])) {
                if (is_a($attributes[$this->genderTax], 'WC_Product_Attribute')) {
                    $terms = $attributes[$this->genderTax]->get_terms();
                } else {
                    $terms = wp_get_object_terms($post->ID, $this->genderTax);
                }
                $names = wp_list_pluck($terms, 'name');
                $this->productsList[$i]['gender'] = implode('|', $names);
            } elseif (isset($parent_attributes[$this->genderTax])) {
                if (is_a($parent_attributes[$this->genderTax], 'WC_Product_Attribute')) {
                    $terms = $parent_attributes[$this->genderTax]->get_terms();
                } else {
                    $terms = wp_get_object_terms($post->ID, $this->genderTax);
                }
                $names = wp_list_pluck($terms, 'name');
                $this->productsList[$i]['size'] = implode('|', $names);
            }
        }

        if ($this->colorTax) {
            if (in_array('attribute_'. $this->colorTax, array_keys($variations_attributes))) {
                $term_slug = $variations_attributes['attribute_'. $this->colorTax];
                $term = get_term_by('slug', $term_slug, $this->colorTax);

                if ($term) {
                    $this->productsList[$i]['colors'] = $term->name;
                    $this->productsList[$i]['title'] .= ' - '. $term->name;
                }
            } elseif (isset($attributes[$this->colorTax])) {
                if (is_a($attributes[$this->colorTax], 'WC_Product_Attribute')) {
                    $terms = $attributes[$this->colorTax]->get_terms();
                } else {
                    $terms = wp_get_object_terms($post->ID, $this->colorTax);
                }
                $names = wp_list_pluck($terms, 'name');
                $this->productsList[$i]['colors'] = implode('|', $names);
            } elseif (isset($parent_attributes[$this->colorTax])) {
                if (is_a($parent_attributes[$this->colorTax], 'WC_Product_Attribute')) {
                    $terms = $parent_attributes[$this->colorTax]->get_terms();
                } else {
                    $terms = wp_get_object_terms($post->ID, $this->colorTax);
                }
                $names = wp_list_pluck($terms, 'name');
                $this->productsList[$i]['colors'] = implode('|', $names);
            }
        }

        /**
         * Overwrite some settings with special products
         */
        if ($type == 'external') {
            $getLink = new \WC_Product_External($post->ID);
            $link = $getLink->get_product_url();
            $this->productsList[$i]['link'] = $link;
        }
    }

    /**
     * Get formatted image url
     *
     * @param $url
     * @return bool|string
     */
    public function get_formatted_url($url = '')
    {
        if (!empty($url)) {
            if (substr(trim($url), 0, 4) === 'http' || substr(trim($url), 0, 3) === 'ftp' || substr(trim($url), 0, 4) === 'sftp') {
                return rtrim($url, '/');
            } else {
                $base = get_site_url();
                $url = $base . $url;

                return rtrim($url, '/');
            }
        }

        return $url;
    }

    /**
     * Get formatted product date
     *
     * @param $id
     * @param $name
     * @return bool|string
     */
    public function get_date($id, $name)
    {
        $date = $this->getAttributeValue($id, $name);
        if ($date) {
            return date('Y-m-d', $date);
        }

        return false;
    }

    /**
     * Get formatted product quantity
     *
     * @param $id
     * @param $name
     * @return bool|mixed
     */
    public function get_quantity($id, $name)
    {
        $qty = $this->getAttributeValue($id, $name);
        if ($qty) {
            return $qty + 0;
        }

        return '0';
    }

    /**
     * Get the Price of a Product
     *
     * @param $product
     */
    public function get_regular_price($product)
    {
        if (is_a($product, 'WC_Product_Variable')) {
            $price = $product->get_variation_regular_price();
        } else {
            $price = $product->get_regular_price();
        }

        if (class_exists('woocommerce_wpml') && wcml_is_multi_currency_on()) {
            $price = $this->get_wcml_price($price, $product, get_woocommerce_currency(), '_regular_price');
        }

        return wc_get_price_including_tax($product, ['price' => $price]);
    }

    /**
     * Get the Sale price of a Product
     *
     * @param [type] $product
     */
    public function get_sale_price($product)
    {
        // Check if the product is actually on sale (WooCommerce handles date validation)
        if (!$product->is_on_sale()) {
            // Not on sale, return regular price
            return $this->get_regular_price($product);
        }

        $price = '';
        if (is_a($product, 'WC_Product_Variable') && $product->get_variation_sale_price()) {
            $price = $product->get_variation_sale_price();
        }

        if (class_exists('woocommerce_wpml') && wcml_is_multi_currency_on()) {
            $price = $this->get_wcml_price($price, $product, get_woocommerce_currency(), '_sale_price');
        }

        if (!$price) {
            $price = $product->get_sale_price() ? $product->get_sale_price() : $product->get_regular_price();
        }

        if (class_exists('\Wdr\App\Controllers\ManageDiscount')) {
            $manage_discount_class = new \Wdr\App\Controllers\ManageDiscount();
            $discounted_price = $manage_discount_class->getProductSalePrice($product->get_price(), $product);

            if ($discounted_price == $price) {
                return wc_get_price_including_tax($product, ['price' => $price]);
            } else {
                return $discounted_price;
            }
        }

        return wc_get_price_including_tax($product, ['price' => $price]);
    }


    public function get_wcml_price($price, $product, $currency, $type = null)
    {
        // Skip conversion if currency is the default WooCommerce currency
        $default_currency = get_option('woocommerce_currency');
        if ($currency === $default_currency) {
            return (float) $price;
        }

        $lang = apply_filters('wpml_default_language', '');
        $originalId = apply_filters('wpml_object_id', $product->get_id(), 'any', true, $lang);

        global $woocommerce_wpml;
        if (is_a($product, 'WC_Product_Variable')) {
            $children = $product->get_children();
            $all_prices = [];
            foreach ($children as $child) {
                $child_product = wc_get_product($child);
                $originalId = apply_filters('wpml_object_id', $child_product->get_id(), 'any', true, $lang);
                if (get_post_meta($originalId, '_wcml_custom_prices_status', true)) {
                    $prices = $woocommerce_wpml->multi_currency->custom_prices->get_product_custom_prices($originalId, $currency);
                    if (! empty($prices)) {
                        if (is_null($type)) {
                            return $prices['_price'];
                        }
                        if (array_key_exists($type, $prices)) {
                            $all_prices[] = $prices[ $type ];
                        } else {
                            $all_prices[] = $prices['_price'];
                        }
                    }
                } else {
                    $currencies = $woocommerce_wpml->multi_currency->currencies;
                    if (array_key_exists($currency, $currencies)) {
                        $price = ((float) $price * (float) $currencies[ $currency ]['rate']);
                        $all_prices[] = $woocommerce_wpml->multi_currency->prices->apply_rounding_rules($price, $currency);
                    }
                }
            }

            if (!empty($all_prices)) {
                $price = min($all_prices);
            }
        } else {
            if (get_post_meta($originalId, '_wcml_custom_prices_status', true)) {
                $prices = $woocommerce_wpml->multi_currency->custom_prices->get_product_custom_prices($originalId, $currency);
                if (! empty($prices)) {
                    if (is_null($type)) {
                        return (float) $prices['_price'];
                    }
                    if (array_key_exists($type, $prices)) {
                        return (float) $prices[ $type ];
                    } else {
                        return (float) $prices['_price'];
                    }
                }
            } else {
                $currencies = $woocommerce_wpml->multi_currency->currencies;
                if (array_key_exists($currency, $currencies)) {
                    $price = ((float) $price * (float) $currencies[ $currency ]['rate']);
                    $price = $woocommerce_wpml->multi_currency->prices->apply_rounding_rules($price, $currency);
                }
            }
        }


        return (float) $price;
    }

    /**
     * Retrieve a post's terms as a list with specified format.
     *
     * @param int $id Post ID.
     * @param string $taxonomy Taxonomy name.
     * @param string $before Optional. Before list.
     * @param string $sep Optional. Separate items using this.
     * @param string $after Optional. After list.
     *
     * @return string|false|WP_Error A list of terms on success, false if there are no terms, WP_Error on failure.
     */
    public function get_product_term_list($id, $taxonomy, $before = '', $sep = ',', $after = '')
    {
        $terms = get_the_terms($id, $taxonomy);

        if (is_wp_error($terms)) {
            return $terms;
        }

        if (empty($terms)) {
            return false;
        }

        $links = [];

        foreach ($terms as $term) {
            $links[] = $term->name;
        }
        ksort($links);

        return $before . join($sep, $links) . $after;
    }

    /**
     * Return additional image URLs
     *
     * @param array $imgIds
     * @return array
     */
    public function additionalImages($imgIds)
    {
        $images = [];
        if (count($imgIds)) {
            foreach ($imgIds as $key => $value) {
                if ($key < 9) {
                    $images[$key] = wp_get_attachment_url($value);
                }
            }

            return $images;
        }

        return false;
    }

    /**
     * Give space to availability text
     *
     * @param bool $param
     *
     * @return string
     */
    public function availability($param = false)
    {
        if ($param) {
            if ($param == 'instock') {
                return 'in stock';
            } elseif ($param == 'outofstock') {
                return 'out of stock';
            }
        }

        return 'out of stock';
    }

    /**
     * Get Product Attribute Value
     *
     * @param $id
     * @param $name
     *
     * @return mixed
     */
    public function getAttributeValue($id, $name)
    {
        if (strpos($name, 'attribute_pa') !== false) {
            $taxonomy = str_replace('attribute_', '', $name);
            $meta = get_post_meta($id, $name, true);
            $term = get_term_by('slug', $meta, $taxonomy);

            return $term->name;
        } else {
            return get_post_meta($id, $name, true);
        }
    }

    /**
     * Get Sale price effective date
     *
     * @param $id
     * @param $name
     * @return string
     */
    public function sale_price_effective_date($id, $name)
    {
        return ($date = $this->getAttributeValue($id, $name)) ? date_i18n('Y-m-d', $date) : '';
    }

    private function get_ean($product, $i)
    {
        global $wpdb;

        // WooCommerce
        if (method_exists($product, 'get_global_unique_id')) {
            $ean = $product->get_global_unique_id();
            if ($ean && !empty($ean)) {
                $this->productsList[$i]['ean'] = $ean;
                return;
            }
        }

        // Plugin
        $ean = get_post_meta($product->get_id(), 'hwp_product_gtin', true);
        if ($ean && !empty($ean)) {
            $this->productsList[$i]['ean'] = $ean;
            return;
        }

        // Plugin
        $ean = get_post_meta($product->get_id(), '_wpm_gtin_code', true);
        if ($ean && !empty($ean)) {
            $this->productsList[$i]['ean'] = $ean;
            return;
        }

        // Just try to find any meta_key with ean in it
        if (!isset($this->productsList[$i]['ean']) || empty($this->productsList[$i]['ean'])) {
            $ean_keys = $wpdb->get_col("SELECT meta_key FROM {$wpdb->postmeta} WHERE meta_key like '%ean' AND post_id = {$product->get_id()}");
            if (count($ean_keys)) {
                foreach ($ean_keys as $key) {
                    $ean = get_post_meta($product->get_id(), $key, true);
                    if ($ean) {
                        $this->productsList[$i]['ean'] = $ean;
                        return;
                    }
                }
            }
        }

        // Just try to find any meta_key with gtin in it
        if (!isset($this->productsList[$i]['ean']) || empty($this->productsList[$i]['ean'])) {
            $gtin_keys = $wpdb->get_col("SELECT meta_key FROM {$wpdb->postmeta} WHERE meta_key like '%gtin%' AND post_id = {$product->get_id()}");
            if (count($gtin_keys)) {
                foreach ($gtin_keys as $key) {
                    $ean = get_post_meta($product->get_id(), $key, true);
                    if ($ean) {
                        $this->productsList[$i]['ean'] = $ean;
                        return;
                    }
                }
            }
        }

        // Try to find EAN in product attributes
        if (!isset($this->productsList[$i]['ean']) || empty($this->productsList[$i]['ean'])) {
            $attributes = $product->get_attributes();
            if (count($attributes)) {
                foreach ($attributes as $key => $attribute) {
                    if (strpos($key, 'ean') !== false) {
                        $ean = $product->get_attribute($key);
                        if ($ean) {
                            $this->productsList[$i]['ean'] = $ean;
                            return;
                        }
                    } elseif (strpos($key, 'gtin') !== false) {
                        $ean = $product->get_attribute($key);
                        if ($ean) {
                            $this->productsList[$i]['ean'] = $ean;
                            return;
                        }
                    }
                }
            }
        }
    }

    private function get_description($product)
    {
        if (Setting::get('feed_descriptions_field') == 'post_excerpt') {
            $description = $product->get_short_description();
        } elseif (Setting::get('feed_descriptions_field') == 'custom_content') {
            $description = get_post_meta($product->get_id(), 'partner_ads_content_field', true);
        } else {
            $description = $product->get_description();
        }

        $description = apply_filters('partner_ads_product_description', strip_tags($description), $product);

        if (empty($description) && $product->get_parent_id()) {
            $parent = wc_get_product($product->get_parent_id());
            $description = $this->get_description($parent);
        }

        return $description;

    }

    private function set_common_product_data($product, $i, $language)
    {
        $post = get_post($this->parentID);
        $mainImage = apply_filters('partner-ads-woocommerce-feed-product-image', wp_get_attachment_url($product->get_image_id()), $product);


        $link = get_permalink($post->ID);

        $this->productsList[$i]['id'] = $product->get_id();
        $this->productsList[$i]['title'] = $product->get_title();
        $this->productsList[$i]['description'] = $this->get_description($product);
        $this->productsList[$i]['category'] = $this->get_product_term_list($post->ID, 'product_cat', '', ' > ');// $this->categories($this->parentID);//TODO
        $this->productsList[$i]['link'] = $link;
        $this->productsList[$i]['image'] = $this->get_formatted_url($mainImage);
        $this->productsList[$i]['sku'] = $product->get_sku();
        $this->productsList[$i]['availability'] = $this->availability($product->get_stock_status());
        $this->productsList[$i]['quantity'] = $this->get_quantity($product->get_id(), '_stock');
        $this->productsList[$i]['price'] = $this->get_regular_price($product);
        $this->productsList[$i]['sale_price'] = $this->get_sale_price($product);
        $this->get_ean($product, $i);

        if (function_exists('wpml_language_is_active_filter')) {
            if (Setting::get($language . '_shipping') != '') {
                $this->productsList[$i]['shipping'] = Setting::get($language .'_shipping');
            }
        } else {
            if (Setting::get('shipping') != '') {
                $this->productsList[$i]['shipping'] = Setting::get('shipping');
            }
        }

        if (function_exists('wpml_language_is_active_filter')) {
            if (Setting::get($language . '_deliverytime') != '') {
                $this->productsList[$i]['deliverytime'] = Setting::get($language .'_deliverytime');
            }
        } else {
            if (Setting::get('delivery') != '') {
                $this->productsList[$i]['deliverytime'] = Setting::get('delivery');
            }
        }
    }
}
