<?php

namespace PartnerAdsWooCommerce;

use PartnerAdsWooCommerce\Settings\Setting;

class PartnerAds
{
    protected $services = [];

    /**
     * @var
     */
    public $pluginPath;

    /**
     * @var
     */
    public $pluginUrl;

    /**
     * @var
     */
    public $file;

    /**
     * Init the plugin
     */
    public function init()
    {
        $this->services['tracker'] = new Tracking\Tracker();
        $this->services['feed_controller'] = new Feed\Controller;

        if (is_admin()) {
            $this->services['script_controller'] = new Admin\ScriptsController;
            $this->services['product_meta_controller'] = new Admin\ProductMetaController();
        }

        add_action('plugins_loaded', [$this, 'initWooCommerceIntegration']);
        add_action('plugins_loaded', [$this, 'loadPluginTextDomain']);

        add_action('wp_enqueue_scripts', [$this, 'enqueueFrontendScript']);

        add_action('wp_scheduled_delete', [$this, 'onActivatePlugin'], 25);
        register_activation_hook($this->file, [$this, 'onActivatePlugin']);
        register_deactivation_hook($this->file, [$this, 'onDeactivatePlugin']);
    }

    /**
     * Add WooCommerce Settings Integrations
     * @param $integrations
     * @return array
     */
    public function addWooCommerceIntegration($integrations)
    {
        $integrations[] = 'PartnerAdsWooCommerce\Settings\PartnerAds';

        return $integrations;
    }

    public function loadPluginTextDomain()
    {
        load_plugin_textdomain(
            'partner-ads-woocommerce',
            false,
            dirname(plugin_basename($this->file)) . '/languages'
        );
    }

    public function enqueueFrontendScript()
    {
        wp_enqueue_script('partner-ads-woocommerce', PARTNER_ADS_WOOCOMMERCE_PLUGIN_URL . 'assets/js/public.js', [], '1.35', true);
        wp_localize_script('partner-ads-woocommerce', 'partnerAdsWooCommerce', [
            'update_url_with_variable' => Setting::get('update_url_with_variable'),
        ]);
    }

    public function onActivatePlugin()
    {
        if (!wp_next_scheduled('partner_ads_regenerate_feed_twice_daily')) {
            wp_schedule_event(time(), 'twicedaily', 'partner_ads_regenerate_feed_twice_daily');
        }
    }

    public function onDeactivatePlugin()
    {
        wp_clear_scheduled_hook('partner_ads_regenerate_feed_twice_daily');
        delete_option('partner-ads-generating_feed');
    }

    /**
     * Init the WooCommerce Settings Integration
     */
    public function initWooCommerceIntegration()
    {
        // Checks if WooCommerce is installed.
        if (class_exists('WC_Integration')) {
            add_filter('woocommerce_integrations', [$this, 'addWooCommerceIntegration']);
        }
    }
}
