<?php

namespace PartnerAdsWooCommerce\Settings;

use PartnerAdsWooCommerce\Util\Log;

class PartnerAds extends \WC_Integration
{
    /**
     * Tracking constructor.
     */
    public function __construct()
    {
        $this->id = 'partner-ads';
        $this->method_title = __('Partner-Ads', 'partner-ads-woocommerce');
        $this->method_description = __('', 'partner-ads');

        // Load the settings.
        $this->init_form_fields();
        $this->init_settings();

        // Define user set variables.
        $this->settings = $this->get_option($this->get_option_key(), []);

        // Actions.
        add_action('woocommerce_update_options_integration_' . $this->id, [$this, 'process_admin_options']);
    }

    /**
     * Initialize integration settings form fields.
     */
    public function init_form_fields()
    {
        if (!function_exists('wpml_language_is_active_filter')) {
            $this->form_fields['country'] = [
                'title' => __('Partner-Ads Country', 'partner-ads-woocommerce'),
                'type' => 'select',
                'options' => [
                    'denmark' => __('Denmark', 'partner-ads-woocommerce'),
                    'sweden' => __('Sweden', 'partner-ads-woocommerce'),
                    'norway' => __('Norway', 'partner-ads-woocommerce'),
                ],
                'default' => 'salg',
                'update' => true,
            ];
        }

        $this->form_fields['tracking_title'] = [
            'title' => __('Tracking', 'partner-ads-woocommerce'),
            'type' => 'title',
            'update' => false,
        ];

        $this->form_fields['program_type'] = [
            'title' => __('Program Type', 'partner-ads-woocommerce'),
            'type' => 'select',
            'options' => [
                'salg' => __('Sale', 'partner-ads-woocommerce'),
                'lead' => __('Lead', 'partner-ads-woocommerce'),
            ],
            'default' => 'salg',
            'update' => true,
        ];

        if (function_exists('wpml_language_is_active_filter')) {
            $showing = false;

            if (
                wpml_language_is_active_filter('', 'da')
                ||
                Setting::get('fallback_country') == 'da'
            ) {
                $showing = true;
                $this->form_fields['da_program_id'] = [
                    'title' => __('Denmark Program Id', 'partner-ads-woocommerce'),
                    'type' => 'text',
                    'update' => true,
                ];
            }

            if (
                wpml_language_is_active_filter('', 'no')
                ||
                Setting::get('fallback_country') == 'no'
            ) {
                $showing = true;
                $this->form_fields['no_program_id'] = [
                    'title' => __('Norway Program Id', 'partner-ads-woocommerce'),
                    'type' => 'text',
                    'update' => true,
                ];
            }

            if (
                wpml_language_is_active_filter('', 'sv')
                || wpml_language_is_active_filter('', 'se')
                || Setting::get('fallback_country') == 'sv'
            ) {
                $showing = true;
                $this->form_fields['sv_program_id'] = [
                    'title' => __('Sweden Program Id', 'partner-ads-woocommerce'),
                    'type' => 'text',
                    'update' => true,
                ];
            }

            if (
                !$showing
                ||
                !empty(Setting::get('fallback_country'))
            ) {
                $this->form_fields['fallback_country'] = [
                    'title' => __('Fallback Country', 'partner-ads-woocommerce'),
                    'type' => 'select',
                    'options' => [
                        '' => __('Choose', 'partner-ads-woocommerce'),
                        'da' => __('Denmark', 'partner-ads-woocommerce'),
                        'sv' => __('Sweden', 'partner-ads-woocommerce'),
                        'no' => __('Norway', 'partner-ads-woocommerce'),
                    ],
                    'default' => '',
                    'update' => true,
                ];
            }
        } else {
            $this->form_fields['program_id'] = [
                'title' => __('Program Id', 'partner-ads-woocommerce'),
                'type' => 'text',
                'update' => true,
            ];
        }

        $this->form_fields['tracking_coupons'] = [
            'title' => __('Enable Coupon tracking', 'partner-ads-woocommerce'),
            'type' => 'checkbox',
            'label' => __('Enable this to send coupon to Partner-ads', 'partner-ads-woocommerce'),
            'default' => 'yes',
            'update' => true,
        ];

        $this->form_fields['feed_title'] = [
            'title' => __('Feed', 'partner-ads-woocommerce'),
            'type' => 'title',
            'update' => false,
        ];

        $this->form_fields['disable_feed'] = [
            'title' => __('Disable auto feed generation', 'partner-ads-woocommerce'),
            'type' => 'checkbox',
            'label' => __('Disable the automatic feed generation', 'partner-ads-woocommerce'),
            'default' => 'no',
            'update' => true,
            'description' => __('You will still be able to manually create a feed from this settings page', 'partner-ads-woocommerce'),
        ];

        $this->form_fields['feed_enabled_descriptions'] = [
            'title' => __('Enable Descriptions', 'partner-ads-woocommerce'),
            'type' => 'checkbox',
            'label' => __('Enable to add descriptions to the feed', 'partner-ads-woocommerce'),
            'default' => 'yes',
            'update' => true,
        ];

        $this->form_fields['feed_descriptions_field'] = [
            'title' => __('Descriptions field', 'partner-ads-woocommerce'),
            'type' => 'select',
            'options' => [
                'post_content' => __('Full description', 'partner-ads-woocommerce'),
                'post_excerpt' => __('Short description', 'partner-ads-woocommerce'),
                'custom_content' => __('Extra field', 'partner-ads-woocommerce'),
            ],
            'default' => 'post_content',
            'update' => true,
        ];

        $this->form_fields['feed_brand'] = [
            'title' => __('Brand Attribute', 'partner-ads-woocommerce'),
            'type' => 'select',
            'default' => '0',
            'options' => $this->getAttributeTaxonomies('feed_brand'),
            'update' => true,
            'description' => __('If your shop are using a Brand attribute you can choose the Attribute here', 'partner-ads-woocommerce'),
        ];

        $this->form_fields['feed_brand_custom'] = [
            'title' => __('Custom Brand Attribute', 'partner-ads-woocommerce'),
            'type' => 'text',
            'default' => '0',
            'update' => true,
            'description' => __('Write the custom attribute you have written in Products', 'partner-ads-woocommerce'),
        ];

        $this->form_fields['feed_size'] = [
            'title' => __('Size Attribute', 'partner-ads-woocommerce'),
            'type' => 'select',
            'default' => '0',
            'options' => $this->getAttributeTaxonomies('feed_size'),
            'update' => true,
            'description' => __('If your shop are using a Size attribute you can choose the Attribute here', 'partner-ads-woocommerce'),
        ];

        $this->form_fields['feed_size_custom'] = [
            'title' => __('Custom Size Attribute', 'partner-ads-woocommerce'),
            'type' => 'text',
            'default' => '0',
            'update' => true,
            'description' => __('Write the custom attribute you have written in Products', 'partner-ads-woocommerce'),
        ];

        $this->form_fields['feed_color'] = [
            'title' => __('Color Attribute', 'partner-ads-woocommerce'),
            'type' => 'select',
            'default' => '0',
            'options' => $this->getAttributeTaxonomies('feed_color'),
            'update' => true,
            'description' => __('If your shop are using a Color attribute you can choose the Attribute here', 'partner-ads-woocommerce'),
        ];

        $this->form_fields['feed_color_custom'] = [
            'title' => __('Custom Color Attribute', 'partner-ads-woocommerce'),
            'type' => 'text',
            'default' => '0',
            'update' => true,
            'description' => __('Write the custom attribute you have written in Products', 'partner-ads-woocommerce'),
        ];

        $this->form_fields['feed_gender'] = [
            'title' => __('Gender Attribute', 'partner-ads-woocommerce'),
            'type' => 'select',
            'default' => '0',
            'options' => $this->getAttributeTaxonomies('feed_gender'),
            'update' => true,
            'description' => __('If your shop are using a Gender attribute you can choose the Attribute here', 'partner-ads-woocommerce'),
        ];

        $this->form_fields['feed_gender_custom'] = [
            'title' => __('Custom Gender Attribute', 'partner-ads-woocommerce'),
            'type' => 'text',
            'default' => '0',
            'update' => true,
            'description' => __('Write the custom attribute you have written in Products', 'partner-ads-woocommerce'),
        ];

        $this->form_fields['feed_add_variants'] = [
            'title' => __('Add variants as products', 'partner-ads-woocommerce'),
            'label' => __('Enable', 'partner-ads-woocommerce'),
            'type' => 'checkbox',
            'default' => '',
            'update' => true,
            'description' => __('If you enabled this all the variants will be added as products.', 'partner-ads-woocommerce'),
        ];


        $this->form_fields['feed_include_zero_prices'] = [
            'title' => __('Include Products with No Cost', 'partner-ads-woocommerce'),
            'label' => __('Enable', 'partner-ads-woocommerce'),
            'type' => 'checkbox',
            'default' => '',
            'update' => true,
            'description' => __('If you enabled this all the products with no cost will be included in the feed.', 'partner-ads-woocommerce'),
        ];

        $this->form_fields['feed_include_out_of_stock'] = [
            'title' => __('Include Products Out of Stock', 'partner-ads-woocommerce'),
            'label' => __('Enable', 'partner-ads-woocommerce'),
            'type' => 'checkbox',
            'default' => '',
            'update' => true,
            'description' => __('If you enabled this all the products out of stock will be included in the feed.', 'partner-ads-woocommerce'),
        ];

        if (function_exists('wpml_language_is_active_filter')) {
            if (wpml_language_is_active_filter('', 'da')) {
                $this->form_fields['da_shipping'] = [
                    'title' => __('Danish Shipping', 'partner-ads-woocommerce'),
                    'type' => 'text',
                    'update' => true,
                    'description' => __('This will be added to your feed as a shipping price for the danish feed', 'partner-ads-woocommerce'),
                ];
            }

            if (wpml_language_is_active_filter('', 'no')) {
                $this->form_fields['no_shipping'] = [
                    'title' => __('Norway Shipping', 'partner-ads-woocommerce'),
                    'type' => 'text',
                    'update' => true,
                    'description' => __('This will be added to your feed as a shipping price for the norwegian feed', 'partner-ads-woocommerce'),
                ];
            }

            if (wpml_language_is_active_filter('', 'sv') || wpml_language_is_active_filter('', 'se')) {
                $this->form_fields['sv_shipping'] = [
                    'title' => __('Sweden Shipping', 'partner-ads-woocommerce'),
                    'type' => 'text',
                    'update' => true,
                    'description' => __('This will be added to your feed as a shipping price for the swedish feed', 'partner-ads-woocommerce'),
                ];
            }
        } else {
            $this->form_fields['shipping'] = [
                'title' => __('Shipping', 'partner-ads-woocommerce'),
                'type' => 'text',
                'update' => true,
                'description' => __('This will be added to your feed as a shipping price', 'partner-ads-woocommerce'),
            ];
        }

        if (function_exists('wpml_language_is_active_filter')) {
            if (wpml_language_is_active_filter('', 'da')) {
                $this->form_fields['da_deliverytime'] = [
                    'title' => __('Danish Delivery time', 'partner-ads-woocommerce'),
                    'type' => 'text',
                    'update' => true,
                    'description' => __('This will be added to your feed as a delivery time for the danish feed', 'partner-ads-woocommerce'),
                ];
            }

            if (wpml_language_is_active_filter('', 'no')) {
                $this->form_fields['no_deliverytime'] = [
                    'title' => __('Norway Delivery time', 'partner-ads-woocommerce'),
                    'type' => 'text',
                    'update' => true,
                    'description' => __('This will be added to your feed as a delivery time for the norwegian feed', 'partner-ads-woocommerce'),
                ];
            }

            if (wpml_language_is_active_filter('', 'sv') || wpml_language_is_active_filter('', 'se')) {
                $this->form_fields['sv_deliverytime'] = [
                    'title' => __('Sweden Delivery time', 'partner-ads-woocommerce'),
                    'type' => 'text',
                    'update' => true,
                    'description' => __('This will be added to your feed as a delivery time for the swedish feed', 'partner-ads-woocommerce'),
                ];
            }
        } else {
            $this->form_fields['delivery'] = [
                'title' => __('Delivery time', 'partner-ads-woocommerce'),
                'type' => 'text',
                'update' => true,
                'description' => __('This will be added to your feed as a delivery time', 'partner-ads-woocommerce'),
            ];
        }

        if (function_exists('wpml_language_is_active_filter')) {
            if (
                wpml_language_is_active_filter('', 'da')
                ||
                Setting::get('fallback_country') == 'da'
            ) {
                $this->form_fields['da_feed'] = [
                    'title' => __('Denmark feed link', 'partner-ads-woocommerce'),
                    'type' => 'text',
                    'disabled' => true,
                    'update' => false,
                    'description' => __('Copy this link and send it to Partner-ads <button class="button js-copy-partner-ads-feed-link">Clik to copy</button>', 'partner-ads-woocommerce'),
                ];
            }

            if (
                wpml_language_is_active_filter('', 'no')
                ||
                Setting::get('fallback_country') == 'no'
            ) {
                $this->form_fields['no_feed'] = [
                    'title' => __('Norway feed link', 'partner-ads-woocommerce'),
                    'type' => 'text',
                    'disabled' => true,
                    'update' => false,
                    'description' => __('Copy this link and send it to Partner-ads <button class="button js-copy-partner-ads-feed-link">Clik to copy</button>', 'partner-ads-woocommerce'),
                ];
            }

            if (
                wpml_language_is_active_filter('', 'sv')
                || wpml_language_is_active_filter('', 'se')
                || Setting::get('fallback_country') == 'sv'
            ) {
                $this->form_fields['sv_feed'] = [
                    'title' => __('Sweden feed link', 'partner-ads-woocommerce'),
                    'type' => 'text',
                    'disabled' => true,
                    'update' => false,
                    'description' => __('Copy this link and send it to Partner-ads <button class="button js-copy-partner-ads-feed-link">Clik to copy</button>', 'partner-ads-woocommerce'),
                ];
            }
        } else {
            $this->form_fields['feed'] = [
                'title' => __('Feed link', 'partner-ads-woocommerce'),
                'type' => 'text',
                'disabled' => true,
                'update' => false,
                'description' => __('Copy this link and send it to Partner-ads <button class="button js-copy-partner-ads-feed-link">Clik to copy</button>', 'partner-ads-woocommerce'),
            ];
        }

        $this->form_fields['update_url_with_variable'] = [
            'title' => __('Keep Partner-Ads variables in the url', 'partner-ads-woocommerce'),
            'type' => 'checkbox',
            'update' => true,
            'description' => '',
        ];

        $this->form_fields['activate_logging'] = [
            'title' => __('Activate Logging', 'partner-ads-woocommerce'),
            'type' => 'checkbox',
            'update' => true,
            'description' => __('Enable this setting if Partner-ads ask you to', 'partner-ads-woocommerce'),
        ];

        if (Setting::get('activate_logging') == 'yes') {
            $this->form_fields['download_loggin_button'] = [
                'title' => __('Download Log', 'partner-ads-woocommerce'),
                'type' => 'button',
                'link' => Log::getUrl(),
                'description' => '',
                'update' => false,
            ];
        }

        $this->form_fields['feed_last_update'] = [
            'title' => __('Last Updated', 'partner-ads-woocommerce'),
            'type' => 'text',
            'disabled' => true,
            'update' => false,
        ];

        $this->form_fields['feed_generate_button'] = [
            'title' => __('Generate Feed', 'partner-ads-woocommerce'),
            'type' => 'button',
            'link' => add_query_arg('action', 'partner_ads_generate_feed', admin_url('admin.php')),
            'description' => __('Feed will automatically be updated twice daily, but can be forced to update with this button', 'partner-ads-woocommerce'),
            'update' => false,
        ];
    }

    /**
     * Generate Text Input HTML.
     *
     * @param  mixed $key
     * @param  mixed $data
     * @return string
     */
    public function generate_button_html($key, $data)
    {
        $field_key = $this->get_field_key($key);
        $defaults = [
            'title' => '',
            'disabled' => false,
            'class' => '',
            'css' => '',
            'placeholder' => '',
            'type' => 'text',
            'desc_tip' => false,
            'description' => '',
            'custom_attributes' => [],
        ];

        $data = wp_parse_args($data, $defaults);

        ob_start(); ?>
        <tr valign="top">
            <th scope="row" class="titledesc">

            </th>
            <td class="forminp">
                <fieldset>
                    <legend class="screen-reader-text"><span><?php echo wp_kses_post($data['title']); ?></span></legend>
                    <a
                        href="<?php echo esc_attr($data['link']) ?>"
                        class="button button-primary <?php echo esc_attr($data['class']); ?>"
                        type="<?php echo esc_attr($data['type']); ?>"
                        name="<?php echo esc_attr($field_key); ?>"
                        id="<?php echo esc_attr($field_key); ?>"
                        style="<?php echo esc_attr($data['css']); ?>"
                        placeholder="<?php echo esc_attr($data['placeholder']); ?>"
                        <?php disabled($data['disabled'], true); ?>
                        <?php echo $this->get_custom_attribute_html($data); ?>
                    >
                        <?php echo wp_kses_post($data['title']); ?>
                    </a>
                    <?php echo $this->get_description_html($data); ?>
                </fieldset>
            </td>
        </tr>
        <?php

        return ob_get_clean();
    }

    /**
     * Processes and saves options.
     * If there is an error thrown, will continue to save and validate fields, but will leave the erroring field out.
     * @return bool was anything saved?
     */
    public function process_admin_options()
    {
        $this->init_settings();

        $post_data = $this->get_post_data();
        $oldSettings = get_option($this->get_option_key(), []);

        foreach ($this->get_form_fields() as $key => $field) {
            if ('title' !== $this->get_field_type($field)) {
                try {
                    if (!$this->form_fields[$key]['update']) {
                        $this->settings[$key] = $oldSettings[$key];
                    } else {
                        $this->settings[$key] = $this->get_field_value($key, $field, $post_data);
                    }
                } catch (\Exception $e) {
                    $this->add_error($e->getMessage());
                }
            }
        }

        $return = update_option($this->get_option_key(), apply_filters('woocommerce_settings_api_sanitized_fields_' . $this->id, $this->settings));

        if (($program_id = Setting::get('program_id')) && !get_option('woocommerce_partner-ads_notified', false)) {
            $this->notifyPartnerAds($program_id);
        }

        return $return;
    }

    /**
     * @return array
     */
    private function getAttributeTaxonomies($field)
    {
        $attributes = wc_get_attribute_taxonomies();
        $return = [
            '0' => __("Don't have", 'partner-ads-woocommerce'),
            'custom_attribute' => __("Custom Product Attribute", 'partner-ads-woocommerce'),
        ];

        if (defined('PWB_PLUGIN_FILE') && $field == 'feed_brand') {
            $return['perfect_brands_for_woocommerce'] = 'Perfect Brands for WooCommerce';
        }

        if ($field == 'feed_brand') {
            $return['brand_tax'] = 'Brand Taxonomy';
        }

        foreach ($attributes as $attribute) {
            $name = wc_attribute_taxonomy_name($attribute->attribute_name);
            $return[$name] = $attribute->attribute_label;
        }

        return $return;
    }

    private function notifyPartnerAds($program_id)
    {
        $string = "Partner-ads Woocommerce Plugin aktiveret på: \n";
        $string .= get_bloginfo('name') . "\n";
        $string .= get_bloginfo('siteurl') . "\n";
        $string .= "\n";
        $string .= "Aktiver tracking ved at klikke her: \n";
        $string .= 'https://partner-ads.com/dk/idovfauto.php?kaldok=1&prg=' . $program_id;

        wp_mail(Setting::getCountryEmail(), 'Partner-ads Woocommerce Integration Aktiveret', $string);
        update_option('woocommerce_partner-ads_notified', true);
    }
}
