<?php

namespace PartnerAdsWooCommerce\Tracking;

use PartnerAdsWooCommerce\Settings\Setting;
use PartnerAdsWooCommerce\Util\Log;

class Tracker
{
    /**
     * Tracker constructor.
     */
    public function __construct()
    {
        add_action('template_redirect', [$this, 'addQueriesFromPartnerAdsToCookie']);
        add_action('woocommerce_checkout_update_order_meta', [$this, 'savePartnerAdsQueries'], 10, 1);
        add_action('woocommerce_new_order', [$this, 'savePartnerAdsQueries'], 10, 1);
        // add_action('woocommerce_update_order', [$this, 'savePartnerAdsQueries'], 10, 1);
        add_action('woocommerce_thankyou', [$this, 'notifyPartnerAds'], 10, 1);
        add_action('klarna_after_kco_confirmation', [$this, 'notifyPartnerAds'], 10, 1);
        add_action('woocommerce_order_status_changed', [$this, 'notifyPartnerAdsOnStatusChange'], 10, 1);

        add_filter('BeetleTracking.Utils.WooCommerce.Order.get_params', [$this, 'addBeetleTrackingParams'], 10, 2);
    }

    /**
     * Look for Partner Ads Queries in the $_GET array
     */
    public function addQueriesFromPartnerAdsToCookie()
    {
        if (!empty($_GET)) {
            $queries = [];
            foreach ($_GET as $key => $value) {
                if (
                    preg_match('/^pa-/i', $key)
                    ||
                    preg_match('/^pacid/i', $key)
                ) {
                    $queries[$key] = $value;
                }
            }

            if (!empty($queries)) {
                setcookie('partner-ads-query', base64_encode(json_encode($queries)), time() + 60 * 60 * 24 * 40, '/');

                Log::log('EVENT', json_encode([
                    'function' => 'addQueriesFromPartnerAdsToCookie',
                    'order_id' => '',

                    'caller' => current_filter(),
                    'queries' => $queries
                ]));
            }
        }
    }

    /**
     * Notify Partner Ads about that the sale has been marked completed
     * @param $order_id
     */
    public function notifyPartnerAds($order_id)
    {
        $order = wc_get_order($order_id);

        $coupons = $order->get_coupon_codes();
        $coupons = array_map('strtolower', $coupons);

        if (
            (
                false != ($partnerAdsQuires = $order->get_meta('_partner-ads-queries'))
                ||
                (
                    !empty($coupons)
                    &&
                    (
                        Setting::get('tracking_coupons') == 'yes'
                        ||
                        empty(Setting::get('tracking_coupons'))
                    )
                )
            )
            &&
            !$order->get_meta('_partner-ads-notified')
        ) {
            $partnerid = '';
            $pacid = '';
            $pavc = implode(',', $coupons);

            if (isset($partnerAdsQuires['pa-partnerid'])) {
                $partnerid = $partnerAdsQuires['pa-partnerid'];
                unset($partnerAdsQuires['pa-partnerid']);
            }

            if (isset($partnerAdsQuires['pacid'])) {
                $pacid = $partnerAdsQuires['pacid'];
                unset($partnerAdsQuires['pacid']);
            }

            $fees = $order->get_fees();
            $feeTotal = 0;
            if ($fees) {
                foreach ($fees as $fee) {
                    $feeTotal += $fee->get_total() + $fee->get_total_tax();
                }
            }

            $total = apply_filters('partner-ads-woocommerce-total', $order->get_total() - $order->get_total_shipping() - $order->get_shipping_tax() - $feeTotal);

            $default = [
                'programid' => Setting::get('program_id'),
                'type' => Setting::get('program_type'),
                'ordreid' => $order->get_order_number(),
                'omprsalg' => $total,
                'partnerid' => $partnerid,
                'pacid' => $pacid,
                'varenummer' => 'x',
                'antal' => '1',
                'pavc' => $pavc,
            ];

            $queries = wp_parse_args($default, $partnerAdsQuires);
            $url = add_query_arg($queries, Setting::getCountryUrl() . 'leadtracks2s.php?pw=wc2017trc');

            wp_remote_get($url);
            $order->update_meta_data('_partner-ads-notified', true);
            $order->save();

            Log::log('EVENT', json_encode([
                'function' => 'notifyPartnerAds',
                'order_id' => $order_id,
                'caller' => current_filter(),
                'queries' => $queries,
                'url' => $url,
            ]));
        }
    }

    /**
     * Notify Partner Ads about that the sale has been marked completed
     * @param $order_id
     */
    public function notifyPartnerAdsOnStatusChange($order_id)
    {
        $order = wc_get_order($order_id);

        if (
            $order->get_status() == 'completed'
            &&
            !$order->get_meta('_partner-ads-notified')
        ) {
            $this->notifyPartnerAds($order_id);
        }
    }

    /**
     * Save the Queries from the Cookie when an order is completed
     * @param $order_id
     * @param $data
     */
    public function savePartnerAdsQueries($order_id)
    {
        if (isset($_COOKIE['partner-ads-query'])) {
            $queries = json_decode(base64_decode($_COOKIE['partner-ads-query']), true);
            $order = wc_get_order($order_id);
            $order->update_meta_data('_partner-ads-queries', $queries);
            $order->save();

            if (!headers_sent()) {
                setcookie('partner-ads-query', null, -1);
            }

            Log::log('EVENT', json_encode([
                'function' => 'savePartnerAdsQueries',
                'order_id' => $order_id,
                'caller' => current_filter(),
                'queries' => $queries
            ]));
        }
    }

    public function addBeetleTrackingParams($params, $order)
    {
        $partnerAdsQuires = $order->get_meta('_partner-ads-queries');
        $params['pa-affiliate-salg'] = $partnerAdsQuires
            ? __('Yes', 'partner-ads-woocommerce')
            : __('No', 'partner-ads-woocommerce');


        return $params;
    }
}
